<?php

namespace GoDaddy\WordPress\MWC\Core\Features\Commerce\Backfill\Jobs;

use Exception;
use GoDaddy\WordPress\MWC\Common\Exceptions\WordPressDatabaseException;
use GoDaddy\WordPress\MWC\Common\Repositories\WordPress\TermsRepository;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Catalog\CatalogIntegration;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Catalog\DataStores\CategoryDataStore;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Catalog\Helpers\CategoryEligibilityHelper;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Commerce;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Exceptions\Contracts\CommerceExceptionContract;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Repositories\CategoryMapRepository;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Repositories\SkippedResources\SkippedCategoriesRepository;
use GoDaddy\WordPress\MWC\Core\JobQueue\DataObjects\BatchJobOutcome;
use WP_Term;

/**
 * Backfill product categories job class.
 */
class BackfillProductCategoriesJob extends AbstractBackfillResourceJob
{
    /** @var string unique identifier for the queue.jobs config */
    public const JOB_KEY = 'backfillProductCategories';

    /** @var CategoryMapRepository category map repository */
    protected CategoryMapRepository $categoryMapRepository;

    /** @var CategoryDataStore categories data store */
    protected CategoryDataStore $categoryDataStore;

    public function __construct(CategoryMapRepository $categoryMapRepository, CategoryDataStore $categoryDataStore, SkippedCategoriesRepository $skippedCategoriesRepository)
    {
        $this->categoryMapRepository = $categoryMapRepository;
        $this->categoryDataStore = $categoryDataStore;

        parent::__construct($skippedCategoriesRepository);
    }

    /**
     * {@inheritDoc}
     */
    protected function processBatch() : BatchJobOutcome
    {
        if (CatalogIntegration::hasCommerceCapability(Commerce::CAPABILITY_WRITE)) {
            $localResources = $this->getLocalResources();

            if (empty($localResources)) {
                // no more records to process!
                return $this->makeOutcome();
            }

            $this->createResourcesInPlatform($localResources);
        }

        return $this->makeOutcome();
    }

    /**
     * Makes a {@see BatchJobOutcome} DTO with an accurate `$isComplete` property, based on the number of items found in the current batch.
     *
     * @return BatchJobOutcome
     */
    protected function makeOutcome() : BatchJobOutcome
    {
        return BatchJobOutcome::getNewInstance([
            // we're all done if we just retrieved fewer resources than we asked for
            'isComplete' => $this->getAttemptedResourcesCount() < $this->getJobSettings()->maxPerBatch,
        ]);
    }

    /**
     * Queries for the local term objects.
     *
     * @return WP_Term[]|null
     */
    protected function getLocalResources() : ?array
    {
        $localIds = $this->categoryMapRepository->getUnmappedLocalIds(
            $this->getJobSettings()->maxPerBatch
        );

        if (empty($localIds)) {
            return null;
        }

        /** @var WP_Term[] $terms */
        $terms = CatalogIntegration::withoutReads(function () use ($localIds) {
            return TermsRepository::getTerms([
                'include'    => $localIds,
                'taxonomy'   => CatalogIntegration::PRODUCT_CATEGORY_TAXONOMY,
                'hide_empty' => false,
            ]);
        });

        $this->setAttemptedResourcesCount(count($terms));

        return $terms;
    }

    /**
     * Attempts to create remote terms from the local copies.
     *
     * @param WP_Term[] $localResources
     * @return void
     * @throws WordPressDatabaseException
     */
    protected function createResourcesInPlatform(array $localResources) : void
    {
        foreach ($localResources as $localResource) {
            $this->maybeCreateResourceInPlatform($localResource);
        }
    }

    /**
     * Creates a resource in the platform if it's eligible. Logs ineligible and failed items.
     *
     * @param WP_Term $category
     * @return void
     * @throws WordPressDatabaseException
     */
    protected function maybeCreateResourceInPlatform(WP_Term $category) : void
    {
        try {
            if (! CategoryEligibilityHelper::shouldWriteCategoryToPlatform($category)) {
                throw new Exception('Category not eligible.');
            }

            $this->categoryDataStore->createOrUpdateCategoryInPlatform($category);
        } catch(Exception|CommerceExceptionContract $exception) {
            $this->markLocalResourceAsSkipped($category->term_id);
        }
    }

    /**
     * {@inheritDoc}
     */
    public function getJobKey() : string
    {
        return static::JOB_KEY;
    }
}
