<?php

namespace GoDaddy\WordPress\MWC\Core\Features\Commerce\Catalog\Services;

use Exception;
use GoDaddy\WordPress\MWC\Common\Configuration\Configuration;
use GoDaddy\WordPress\MWC\Common\Exceptions\BaseException;
use GoDaddy\WordPress\MWC\Common\Exceptions\SentryException;
use GoDaddy\WordPress\MWC\Common\Helpers\ArrayHelper;
use GoDaddy\WordPress\MWC\Common\Helpers\TypeHelper;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Catalog\Operations\ListProductsOperation;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Catalog\Providers\DataObjects\ProductAssociation;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Catalog\Services\Contracts\ListProductsServiceContract;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Exceptions\Contracts\CommerceExceptionContract;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Exceptions\MissingRemoteIdsAfterLocalIdConversionException;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Services\Exceptions\CachingStrategyException;

/**
 * Service class to aid in listing products by ID in batches.
 * The maximum we can request by ID at one time is `100`, otherwise the URL becomes too long. This service class batches
 * up the local IDs to perform multiple API requests until we've retrieved them all.
 */
class BatchListProductsByLocalIdService
{
    /** @var ListProductsServiceContract service to list products */
    protected ListProductsServiceContract $listProductsService;

    /**
     * Constructor.
     *
     * @param ListProductsServiceContract $listProductsService
     */
    public function __construct(ListProductsServiceContract $listProductsService)
    {
        $this->listProductsService = $listProductsService;
    }

    /**
     * Gets the maximum number of products we should query in one API request.
     *
     * @return int<1, max>
     */
    protected function getMaximumProductsPerRequest() : int
    {
        return max(TypeHelper::int(Configuration::get('commerce.catalog.api.maxIdsPerRequest'), 100), 1);
    }

    /**
     * Lists products from the API in batches, using the local product IDs.
     *
     * @param int[] $localIds
     * @return ProductAssociation[] results from all batches merged into one array
     */
    public function batchListByLocalIds(array $localIds) : array
    {
        $productAssociations = [];

        foreach (array_chunk($localIds, $this->getMaximumProductsPerRequest()) as $localIdsBatch) {
            try {
                /** @var ProductAssociation[] $productAssociations */
                $productAssociations = ArrayHelper::combine($productAssociations, $this->listBatch($localIdsBatch));
            } catch(MissingRemoteIdsAfterLocalIdConversionException $exception) {
                // we don't need to report this exception to Sentry
            } catch(Exception|CommerceExceptionContract $exception) {
                SentryException::getNewInstance($exception->getMessage(), $exception);
            }
        }

        return $productAssociations;
    }

    /**
     * Lists a batch of products by local IDs.
     *
     * @param int[] $localIds
     * @return ProductAssociation[]
     * @throws MissingRemoteIdsAfterLocalIdConversionException|BaseException|CommerceExceptionContract|CachingStrategyException
     */
    protected function listBatch(array $localIds) : array
    {
        return $this->listProductsService->list(
            ListProductsOperation::seed(['localIds' => $localIds])
        );
    }
}
