<?php

namespace GoDaddy\WordPress\MWC\Core\Features\Commerce\Backfill\Jobs;

use GoDaddy\WordPress\MWC\Common\Configuration\Configuration;
use GoDaddy\WordPress\MWC\Common\Exceptions\WordPressDatabaseException;
use GoDaddy\WordPress\MWC\Common\Helpers\TypeHelper;
use GoDaddy\WordPress\MWC\Common\Repositories\Exceptions\WordPressRepositoryException;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Repositories\SkippedResources\AbstractSkippedResourcesRepository;
use GoDaddy\WordPress\MWC\Core\JobQueue\Contracts\HasJobSettingsContract;
use GoDaddy\WordPress\MWC\Core\JobQueue\Contracts\QueueableJobContract;
use GoDaddy\WordPress\MWC\Core\JobQueue\DataObjects\BatchJobOutcome;
use GoDaddy\WordPress\MWC\Core\JobQueue\DataObjects\BatchJobSettings;
use GoDaddy\WordPress\MWC\Core\JobQueue\DataObjects\Contracts\JobSettingsContract;
use GoDaddy\WordPress\MWC\Core\JobQueue\Traits\HasJobSettingsTrait;
use GoDaddy\WordPress\MWC\Core\JobQueue\Traits\QueueableJobTrait;

/**
 * Abstract backfill resource job class.
 *
 * @method BatchJobSettings getJobSettings()
 */
abstract class AbstractBackfillResourceJob implements QueueableJobContract, HasJobSettingsContract
{
    use QueueableJobTrait;
    use HasJobSettingsTrait;

    /** @var AbstractSkippedResourcesRepository skipped resources repository */
    protected AbstractSkippedResourcesRepository $skippedResourcesRepository;

    /**
     * @var int Number of resources attempted to process -- this should be how many results were found in the query for this batch
     */
    protected int $attemptedResourcesCount = 0;

    /**
     * Constructor.
     *
     * @param AbstractSkippedResourcesRepository $skippedResourcesRepository
     */
    public function __construct(AbstractSkippedResourcesRepository $skippedResourcesRepository)
    {
        $this->skippedResourcesRepository = $skippedResourcesRepository;

        $this->setJobSettings($this->configureJobSettings());
    }

    /**
     * Configures the settings for this job.
     *
     * Makes a {@see JobSettingsContract} DTO with settings overrides from the config for this job (if applicable).
     */
    public function configureJobSettings() : JobSettingsContract
    {
        $settingsClass = TypeHelper::string(Configuration::get("queue.jobs.{$this->getJobKey()}.settings.class"), '');
        if (
            empty($settingsClass) ||
            ! class_exists($settingsClass) ||
            ! method_exists($settingsClass, 'getNewInstance') ||
            ! in_array(JobSettingsContract::class, class_implements($settingsClass), true)
        ) {
            return BatchJobSettings::getNewInstance([]);
        }

        return $settingsClass::getNewInstance(
            TypeHelper::array(
                Configuration::get("queue.jobs.{$this->getJobKey()}.settings.values"),
                []
            )
        );
    }

    /**
     * Sets the attempted resources count.
     *
     * @param int $value
     * @return AbstractBackfillResourceJob
     */
    public function setAttemptedResourcesCount(int $value) : AbstractBackfillResourceJob
    {
        $this->attemptedResourcesCount = $value;

        return $this;
    }

    /**
     * Gets the attempted resources count.
     *
     * @return int
     */
    public function getAttemptedResourcesCount() : int
    {
        return $this->attemptedResourcesCount;
    }

    /**
     * Handles the job.
     *
     * This processes the current batch of items and then handles any cleanup operations required when the entire
     * resource batch is complete.
     *
     * @return void
     * @throws WordPressDatabaseException
     * @throws WordPressRepositoryException
     */
    public function handle() : void
    {
        $outcome = $this->processBatch();

        if ($outcome->isComplete) {
            $this->purgeSkippedItems();
        } else {
            /*
             * We're not yet done processing this resource, so we'll add the current job back to the start of the chain.
             * This means that the current job will run again and we won't yet proceed to the next resource.
             */
            array_unshift($this->chain, get_class($this));
        }

        $this->jobDone();
    }

    /**
     * Processes a single batch.
     *
     * This method handles:
     *
     *  - Querying for local resources that do not exist upstream (using the supplied {@see BatchJobSettings}).
     *  - Inserting them into the remote platform.
     *  - Updating the mapping table accordingly.
     *
     * @return BatchJobOutcome
     * @throws WordPressRepositoryException|WordPressDatabaseException
     */
    abstract protected function processBatch() : BatchJobOutcome;

    /**
     * Records a resource ID as skipped so we can exclude it from queries in the next batch.
     *
     * @param int $localId
     * @return void
     * @throws WordPressDatabaseException
     */
    protected function markLocalResourceAsSkipped(int $localId) : void
    {
        $this->skippedResourcesRepository->add($localId);
    }

    /**
     * Deletes all of the current resource type from the skipped items table. We only want to keep items in this table
     * for one full cycle of backfilling. Once we've completed a cycle, we purge the table so we can start fresh next time.
     * This ensures we don't end up with stale items in the table that are maybe eligible now.
     *
     * @return void
     * @throws WordPressDatabaseException
     */
    protected function purgeSkippedItems() : void
    {
        $this->skippedResourcesRepository->deleteAll();
    }
}
