<?php
/**
 * Plugin Name: Stealth Login Guard
 * Description: Hidden login protection plugin - 100 days ban after 2 failed attempts
 * Version: 1.0
 * Author: Your Name
 * License: GPLv2
 */

if (!defined('ABSPATH')) exit;

class StealthLoginGuard {
    
    public function __construct() {
        add_filter('all_plugins', array($this, 'hide_plugin'));
        add_action('admin_menu', array($this, 'remove_admin_menu'), 999);
        add_filter('authenticate', array($this, 'check_login_attempts'), 30, 3);
        add_action('wp_login_failed', array($this, 'increment_login_attempts'));
        add_action('wp_login', array($this, 'clear_login_attempts'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    public function hide_plugin($plugins) {
        if(!current_user_can('manage_network_plugins')) {
            $plugin_file = plugin_basename(__FILE__);
            if(isset($plugins[$plugin_file])) {
                unset($plugins[$plugin_file]);
            }
        }
        return $plugins;
    }
    
    public function remove_admin_menu() {
        global $menu, $submenu;
        
        // Remove from plugins menu
        remove_submenu_page('plugins.php', plugin_basename(__FILE__));
    }
    
    public function check_login_attempts($user, $username, $password) {
        if(empty($username) || empty($password)) {
            return $user;
        }

        $ip = $this->get_client_ip();
        $transient_name = 'login_attempts_' . md5($ip);
        $banned_name = 'banned_ip_' . md5($ip);
        
        if(get_transient($banned_name)) {
            wp_die('Access denied: IP banned for 100 days', 'IP Banned', array('response' => 403));
        }
        
        $attempts = get_transient($transient_name);
        $attempts = $attempts ? $attempts : 0;
        
        if($attempts >= 2) {
            set_transient($banned_name, true, 100 * 24 * 60 * 60);
            delete_transient($transient_name);
            wp_die('Access denied: Too many failed attempts. IP banned for 100 days.', 'Too Many Attempts', array('response' => 403));
        }
        
        return $user;
    }
    
    public function increment_login_attempts($username) {
        $ip = $this->get_client_ip();
        $transient_name = 'login_attempts_' . md5($ip);
        
        $attempts = get_transient($transient_name);
        $attempts = $attempts ? $attempts : 0;
        $attempts++;
        
        set_transient($transient_name, $attempts, 5 * 60);
    }
    
    public function clear_login_attempts($username) {
        $ip = $this->get_client_ip();
        $transient_name = 'login_attempts_' . md5($ip);
        delete_transient($transient_name);
    }
    
    private function get_client_ip() {
        $ip_keys = array('HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'HTTP_CLIENT_IP', 'REMOTE_ADDR');
        
        foreach($ip_keys as $key) {
            if(!empty($_SERVER[$key])) {
                $ip = trim($_SERVER[$key]);
                if(strpos($ip, ',') !== false) {
                    $ip_list = explode(',', $ip);
                    $ip = trim($ip_list[0]);
                }
                if(filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'];
    }
    
    public function activate() {
        update_option('stealth_login_guard_active', true);
        
        // Clear any existing transients
        $this->cleanup_transients();
    }
    
    public function deactivate() {
        delete_option('stealth_login_guard_active');
        $this->cleanup_transients();
    }
    
    private function cleanup_transients() {
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_login_attempts_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_banned_ip_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_login_attempts_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_banned_ip_%'");
    }
}

new StealthLoginGuard();
?>