from typing import TYPE_CHECKING
import time
from datetime import timedelta, datetime
import sqlite3
from tabulate import tabulate
from cpapis import whmapi1, CpAPIError

if TYPE_CHECKING:
    from disk_cleanup import DiskCleaner


def run_disk_change(cleaner: 'DiskCleaner') -> None:
    """Displays change in disk usage on a per-user basis"""
    date_format = "%b%d%Y"
    today = time.strftime(date_format)
    days = int(cleaner.days)

    try:
        cleaner.logger.debug(
            'action=run_disk_change sqlite3.connect status=connecting to '
            '/etc/rads/quotas.db'
        )
        conn = sqlite3.connect('/etc/rads/quotas.db')
        cur = conn.cursor()
    except sqlite3.Error as e:
        cleaner.logger.error(
            'action=run_disk_change sqlite3.connect error=%s', e
        )
        return

    try:
        cleaner.logger.debug(
            'action=run_disk_change sqlite3.execute '
            'status=creating table %s',
            today,
        )
        cur.execute(f"drop table if exists '{today}'")
        cur.execute(f"create table '{today}' (diskused integer, user text)")
    except sqlite3.Error as e:
        cleaner.logger.error(
            'action=run_disk_change sqlite3.execute error=%s', e
        )
        return

    try:
        cleaner.logger.debug(
            'action=run_disk_change whmapi1 listaccts '
            'status=connecting to whmapi1'
        )
        quota_data = whmapi1('listaccts', {"want": 'user,diskused'}, check=True)
    except CpAPIError as e:
        cleaner.logger.error(
            'action=run_disk_change whmapi1 listaccts error=%s', e
        )
        return

    try:
        cleaner.logger.debug(
            'action=run_disk_change sqlite3.execute '
            'status=populating table %s',
            today,
        )
        for user in quota_data['data']['acct']:
            cur.execute(f"insert into '{today}' values (:diskused,:user)", user)
    except sqlite3.Error as e:
        cleaner.logger.error(
            'action=run_disk_change sqlite3.execute error=%s', e
        )
        return

    conn.commit()

    # Get a list of all the tables in /etc/rads/quotas.db
    try:
        cleaner.logger.debug(
            'action=run_disk_change sqlite3.execute '
            'status=querying /etc/rads/quotas.db'
        )
        cur.execute("SELECT name FROM sqlite_master WHERE type='table'")
        tables = cur.fetchall()
    except sqlite3.Error as exc:
        cleaner.logger.info(
            "action=run_disk_change sqlite3.execute error=%s", exc
        )
        return

    user_days_ago = datetime.strftime(
        datetime.today() - timedelta(days), date_format
    )

    # See if data exists in /etc/rads/quotas.db for the day the user
    # requested to compare against
    if user_days_ago not in [table[0] for table in tables]:
        cleaner.logger.warning(
            'action=run_disk_change user_days_ago '
            'status=unavailable date requested'
        )
        try:
            user_days_ago = tables[len(tables) - 2][0]
        except IndexError as exc:
            cleaner.logger.error(
                'action=run_disk_change user_days_ago error=%s', exc
            )
            return

    # Query /etc/rads/quotas.db to determine 10 users with
    # the highest change in disk usage
    try:
        cleaner.logger.debug(
            'action=run_disk_change sqlite3.execute status=querying '
            '/etc/rads/quotas.db'
        )
        cur.execute(
            'SELECT {0}.user,{0}.diskused,{1}.diskused,'
            '({1}.diskused - {0}.diskused) AS diskchange '
            'FROM {0} INNER JOIN {1} ON {0}.user={1}.user '
            'ORDER BY diskchange '
            'DESC limit 10'.format(user_days_ago, today)
        )
    except sqlite3.OperationalError as e:
        cleaner.logger.error(
            'action=run_disk_change sqlite3.execute error=%s', e
        )
        return

    table = []
    headers = ["User", user_days_ago, today, "Disk Change"]

    cleaner.logger.debug(
        'action=run_disk_change display status=populating table'
    )
    for row in cur:
        table.append(row)

    cleaner.logger.debug('action=run_disk_change display status=display table')
    print(tabulate(table, headers, tablefmt="fancy_grid"))

    try:
        cleaner.logger.debug(
            'action=run_disk_change sqlite3.close status=closing '
            'sqlite3 connection'
        )
        conn.close()
    except sqlite3.Error as e:
        cleaner.logger.error('action=run_disk_change sqlite3.close error=%s', e)
