"""Checks for large, old backup buddy directories"""

from collections import defaultdict
from pathlib import Path
import rads
from ..base import ModuleBase

# the default for backupbuddy is wp-content/uploads/backupbuddy_backups/
# however it's configurable, so scan for its zip files
REGEX = (
    # backup-domain_com-
    r'.*\/backup\-[0-9A-Za-z_]{1,63}_[A-Za-z]+\-'
    # 2022-01-01_04am-
    r'[0-9]{4}_[0-9]{2}_[0-9]{2}-[0-9]{2}_[0-9]{2}[ap]?m?-'
    # full-randomstring.zip
    r'\w+-[0-9a-zA-Z-]+\.zip'
)


class Module(ModuleBase):
    """Checks for large, old backup buddy directories"""

    def run_module(self, homedir):
        backup_list = self.find(homedir, regex=REGEX, mtime="+7", type='f')
        if not backup_list:
            return {}  # no email
        # if total size is < 1GiB, do nothing
        if sum(map(self.calc_bytes, backup_list)) < 2**30:
            self.logger.debug('backups too small, skipping %s', homedir)
            return {}  # no email
        # group by parent directories
        parents: defaultdict[Path, list[Path]] = defaultdict(list)
        for backup in backup_list:
            parents[backup.parent].append(backup)
        paths = ', '.join(map(str, parents))
        self.logger.info('backupbuddy dir(s): %s', paths)
        return {
            'template': 379 if rads.IMH_CLASS == 'hub' else 104,
            'variable_1': paths,
        }
