"""Rads module

.. data:: rads.IMH_ROLE:

    The server's imh_role in salt
    *(str | None)*

.. data:: rads.IMH_CLASS:

    The server's imh_class in salt
    *(str | None)*

.. data:: rads.HAS_CPANEL:

    Whether /usr/local/cpanel/cpsrvd exists as a file
    *(bool)*

.. data:: rads.SECURE_USER:

    The secure user's username
    *(str)*

.. data:: rads.SECURE_FQDN:

    The secure user's fqdn
    *(str)*

.. data:: rads.STAFF_GROUPS:

    IMH employee group names
    *(list)*

.. data:: rads.SYS_USERS:

    System usernames. This also includes ``OUR_RESELLERS`` and ``SECURE_USER``
    *(list)*

.. data:: rads.SYS_MYSQL_USERS:

    System MySQL usernames
    *(list)*

.. data:: rads.OUR_RESELLERS:

    Our "main" resellers' usernames on shared, including root and tier2s
    *(list)*

.. data:: rads.CPANEL_EXTRA_SHARED_IPS:

    Any extra shared ips defined in /etc/reservedipreasons
    *(list)*

.. data:: rads.CPANEL_INMOTION_IPS:

    Any shared ips defined in /var/cpanel/mainips/inmotion
    *(list)*
"""

from pathlib import Path
import platform
import re
from . import _globals

__version__ = '4.3.2'

# pylint: disable=wrong-import-position,global-statement
IMH_ROLE, IMH_CLASS = _globals.read_server_role()
IMH_PROFILE = f"{IMH_ROLE}.{IMH_CLASS}" if IMH_ROLE else None
HAS_CPANEL = Path('/usr/local/cpanel/cpsrvd').is_file()
STAFF_GROUPS = ["ahs", "eng", "tier2c", "tier2s", "tier3"]
SYS_USERS = [
    "tier1",
    "tier1adv",
    "root",
    "bin",
    "daemon",
    "sync",
    "cpanel",
    "mail",
    "halt",
    "shutdown",
    "adm",
    "nobody",
    "news",
    "sshd",
    "named",
    "rpc",
    "ftp",
    "apache",
    "mailnull",
    "mysql",
    "mailman",
    "inmotion",
    "hubhost",
    "tier2s",
    "promgr",
    "nagios",
    "moveuser",
    "system",
]
SYS_MYSQL_USERS = ["system user", "root", "cpanel", "mysql", "replicant"]
OUR_RESELLERS = ["root", "tier2s", "inmotion", "hubhost"]
if IMH_ROLE == 'shared':
    SECURE_FQDN = re.sub('^[a-z-]+', 'secure', platform.node())
    SECURE_USER = _globals.get_secure_user(SECURE_FQDN)
    if SECURE_USER:
        SYS_USERS.append(SECURE_USER)
    CPANEL_EXTRA_SHARED_IPS = _globals.get_cpanel_inmotion_ips()
    CPANEL_INMOTION_IPS = _globals.get_cpanel_extra_shared_ips()
else:
    SECURE_FQDN, SECURE_USER = None, None
    CPANEL_EXTRA_SHARED_IPS, CPANEL_INMOTION_IPS = [], []

from . import color
from . import vz

# General functions (these also need __module__ set to appear in docs)
from ._email import send_email

from ._input import get_keypress, prompt_y_n

from ._locking import LockError, lock, wait_lock

from ._logging import setup_logging, setup_verbosity
from ._logging import LevelFilter, MultilineFormatter

from ._sa import get_cp_usage, SaUsage

from ._fsquota import QuotaCtl, QuotaError, QuotaNoUser, QuotasDisabled

from ._users import CpuserError, UserData, UserDomain
from ._users import get_login, get_homedir, get_primary_domain
from ._users import is_cpuser, all_cpusers, main_cpusers
from ._users import get_owner, is_child, get_children
from ._users import cpuser_safe, cpuser_suspended, whoowns
from ._users import DumbYamlLoader, get_plan

from ._tickets import make_ticket, TicketError

# Backwards compat imports
# pylint: disable=wrong-import-order,cyclic-import
from . import semaphore
