"""Global settings"""

from pathlib import Path
import yaml
from ._yaml import DumbYamlLoader


def read_server_role() -> tuple[None, None] | tuple[str, str]:
    """Read /etc/server.role and return imh_role, imh_class as a tuple of str"""
    try:
        server_role = Path('/etc/server.role').read_text(encoding='ascii')
    except OSError:
        imh_role, imh_class = None, None
    else:
        imh_role, imh_class = server_role.strip().split(':', maxsplit=1)
    return imh_role, imh_class


def get_secure_user(secure_fqdn: str) -> str | None:
    """Get the "secure" shared username from /etc/trueuserdomains"""
    try:
        user_domains: dict = yaml.load(
            Path('/etc/trueuserdomains').read_bytes(),
            DumbYamlLoader,
        )
    except OSError:  # If present, it's root.mail 640
        return None
    if not user_domains:
        return None
    return user_domains.get(secure_fqdn, None)


def get_cpanel_extra_shared_ips() -> list[str]:
    """get any extra shared ips"""
    try:
        ipreasons = Path('/etc/reservedipreasons').read_text('ascii')
    except OSError:
        return []
    ips = []
    for line in ipreasons.splitlines():
        try:
            addr, reason = line.split('=', maxsplit=1)
        except ValueError:
            continue  # blank or malformed line
        if 'mail' in reason.lower():
            ips.append(addr.strip())
    return ips


def get_cpanel_inmotion_ips() -> list[str]:
    """get the shared inmotion ips"""
    try:
        return Path('/var/cpanel/mainips/inmotion').read_text('ascii').split()
    except OSError:
        return []
