"""Convenience functions for imh-icinga2-conf-api

These are for the admin server responsible for configuring Icinga,
not for use by end hosts running NRPE/PMS/etc"""
import requests
import json

class APIResponse(object):
    """Object hold a response from conf_api with accessible attributes
    for .data, .status, and .message"""

    def __init__(self, data):
        """Take a json response from conf_api and build an APIResponse"""
        for key, val in data.iteritems():
            setattr(self, key, val)

    def __repr__(self):
        """Allows repr() on APIResponses"""
        return 'APIResponse({0.__dict__})'.format(self)

class APIServerError(Exception):
    """Error reaching the conf api"""

def conf_api(endpoint, api_user, api_pass, config_master=None, warn_only=False, **kwargs):
    """Convenience function for accessing the Icinga2 Configuration API.
    Will either raise APIServerError or return an APIResponse object which wil have
    .data , .status, and .message attributes.

    If warn_only=False(default) and .status != 0, an APIServerError will be raised instead.

    If conf_api fails to reach the server at all, then APIServerError will be raised
    regardless.

    Any variables which must be passed into the function should be passed as optional
    keywoard args, for example:
    conf_api('list_standards', servertype='vp_node')

    Standard Params:
    `endpoint`: API endpoint to hit, such as 'list_hosts'
    `api_user`: username to sign into the API with
    `api_pass`: password to sign into the API with
    """
    if config_master is None:
        config_master = 'lax-sys-pro-icinga1'
    data = {}
    for key, val in kwargs.iteritems():
        if isinstance(val, dict) or isinstance(val, list):
            data[key] = json.dumps(val)
        else:
            data[key] = val
    try:
        result_raw = requests.post(
            'https://%s.imhadmin.net:9443/%s' % (
                config_master,
                endpoint
            ),
            headers={
                'API_USER': api_user,
                'API_PASS': api_pass
            },
            data=data
        ).text
        resp = APIResponse(json.loads(result_raw))
        if not warn_only and resp.status != 0:
            raise APIServerError(
                resp.message,
                'master={0!r} endpoint={1!r} params={2!r}'.format(
                    config_master, endpoint, data
                )
            )
        return resp
    except (requests.exceptions.RequestException, ValueError) as exc:
        raise APIServerError(
            repr(exc),
            'master={0!r} endpoint={1!r} params={2!r}'.format(
                config_master, endpoint, data
            )
        )

