"""This module serves to allow for API calls to be made to Power Panel and may
be imported in order to do so.

These functions do relatively little input validation in order to keep the
module simple. It is expected that scripts using these functions will perform
their own validation.
"""
import requests
import json
import socket

KEY_FILE = '/etc/pp_api.key'
LEGACY_KEY_DIR = '/etc/pp_api'

def _get_api_key():
    """Returns the key as a string ready to use in requests to
    Power Panel APIs
    """
    key_file = KEY_FILE
    key = ""
    try:
        with open(key_file) as key_data:
            key = "".join(key_data.read().strip().split())
    except IOError:
        pass

    return key

def note_account(user, note, adm_user, type, flagged=0):
    """Note the account using the legacy API

    'user' must be a valid user in Power Panel
    'note' is the note to insert
    'adm_user' is the user to whom the note should be credited
    'type' must be a valid note type, e.g. "Suspension"
    'flagged' should be a 1 or 0, and is interpreted as a boolean

    If the note is added successfully the function will return True,
    otherwise the default return value of None will be returned.
    """

    with open("%s/note_account.key" % LEGACY_KEY_DIR) as key_file:
        key = key_file.readline().strip()

    request_payload = {
            'from': 1,
            'key': key,
            'adm_user': adm_user,
            'user': user,
            'type': type,
            'flagged': flagged,
            'note': note
        }

    headers = {'User-Agent': 'IMH PPv2 API Curl - T3'}
    request = requests.get(
            'https://secure1.inmotionhosting.com/api/insert_note.php',
            params=request_payload,
            headers=headers,
        )
    if 'SUCCESS' in request.text:
        return True

def send_notification(user, template, **variables):
    """Sends a notification to the customer.

    'user' must be a valid user in Power Panel and be present on
    the server where this function is called.
    'template' must be a string or an integer that corresponds to
    an existing Power Panel email template ID.
    'variables' is optional and is used to supply keyword=value
    pairs required by the template being used.

    This function will return the entire response from Power Panel as a dict or
    False if an exception is raised while making the request.

    For full details on the API refer to the documentation referenced below:
    https://trac.imhtech.net/Development/wiki/InMotion/Projects/PowerPanelAPI/CommandReference#notification.send
    """
    url = 'https://secure1.inmotionhosting.com/api/notification/send'
    key = _get_api_key()
    headers = {
            'Content-Type': 'application/json',
            'Authorization': "%s:%s" % (socket.getfqdn(), key),
        }
    request_payload = {
            'cpanelUser': user,
            'template': template
        }

    for key, value in variables.iteritems():
        request_payload[key] = value

    request = requests.post(url, headers=headers, data=json.dumps(request_payload))
    
    try:
        response = json.loads(request.text)
    except Exception:
        return False

    return response

