"""The PowerPanel Python API"""

import json
import requests
import platform
import socket

__all__ = ["PowerPanel", "PowerPanelAuth", "PowerPanelResponse"]

class PowerPanelAuth(requests.auth.AuthBase):
    """Attaches an Authorization header to a PowerPanel request"""
    def __init__(self, key_file_path="/etc/pp_api.key"):
        """Initializes the Auth generator with the key file"""
        self.key_file_path = key_file_path

    def _get_api_key(self):
        """Reads the pp API key and returns the raw key data.

        Raises IOError if the key file does not exist or cannot be read.
        """
        with open(self.key_file_path, 'r') as key_data:
            return "".join(key_data.read().strip().split())

    def __call__(self, r):
        """Modifies request adding the auth header"""
        r.headers['Authorization'] = '{}:{}'.format(
            socket.getfqdn(),
            self._get_api_key()
        )
        return r


class PowerPanel(object):
    """A Pythonic interface to the PowerPanel API.

    Normally, one would use only the "call" method.  The first argument is the
    API name (dot or slash separated), the remaining arguments are keyword
    arguments passed directly to the API as described here:
        https://trac.imhtech.net/Development/wiki/InMotion/Projects
        /PowerPanelAPI/CommandReference
    """
    def __init__(self, key_file_path="/etc/pp_api.key"):
        self.key_file_path = key_file_path
        self.pp_api_base_url = "https://secure1.inmotionhosting.com/api"

    def call(self, api_name, **argument_dict):
        """Makes a call to the powerpanel API by name.

        api_name can be dot or slash separated.  i.e. the following are
        equivalent:
            reporting.provisioning
            reporting/provisioning

        argument_dict is a dictionary of arguments to pass to the API as data.

        See:
            https://trac.imhtech.net/Development/wiki/InMotion/Projects
            /PowerPanelAPI/CommandReference
        """
        api_name = api_name.replace('.', '/')
        headers = {
            'Content-Type': 'application/json',
        }
        try:
            request_object = requests.post(
                "{}/{}".format(self.pp_api_base_url, api_name),
                data=json.dumps(argument_dict),
                headers=headers,
                auth=PowerPanelAuth(self.key_file_path)
            )
        except requests.exceptions.RequestException:
            request_object = None
        return PowerPanelResponse(request_object)

    def __call__(self, api_name, **argument_dict):
        """Wraps the call function, making an instance of this class directly
        callable."""
        return self.call(api_name, **argument_dict)

class PowerPanelResponse(object):
    """The response object from a PowerPanel call.

    Properties:
        status:
            The status code returned by the API.  Status is -1 if no response
            was acquired.
        message:
            The message from the PowerPanel API.  "ERROR: [message]" if status
            is -1.
        data:
            Dictionary containing the data from the API.
        raw_response:
            The actual requests object returned after querying the API.
    """
    def _generate_error(self, message, request_object=None):
        """Sets self properties to handle error with [message]"""
        self.status = -1
        self.message = "ERROR: {}".format(message)
        self.data = {}
        self.raw_response = request_object

    def __init__(self, request_object):
        """Parses a requests object containing the power panel API response and
        populates this object's properties.
        """
        if request_object is None:
            self._generate_error("Could not get response from PowerPanel")
            return
        try:
            json_response = request_object.json()
        except ValueError:
            self._generate_error(
                "Could not parse response from PowerPanel",
                request_object
            )
            return
        try:
            self.status = json_response['status']
            self.message = json_response['message']
            self.data = json_response['data']
            self.raw_response = request_object
        except KeyError:
            self._generate_error(
                "Could not parse response from PowerPanel",
                request_object
            )
