"""Backup Authority API error classes"""

from .sess import Status


class BakAuthError(Exception):
    """Base class raised for errors with the Backup Authority API. All other
    exceptions subclass this, so catching this catches them all

    Args:
        status (Status): error code
        data (str): error message
    """

    __module__ = 'bakauth'

    # pylint: disable=unused-argument,too-many-return-statements
    def __new__(cls, *, status: Status, data: str):
        if cls is not BakAuthError:
            return Exception.__new__(cls)
        if status is Status.REQUEST_EXCEPTION:
            return Exception.__new__(BakAuthDown)
        if status is Status.OKAY:  # impossible
            raise ValueError('0 is not an error code')
        if status is Status.AMP_DOWN:
            return Exception.__new__(AMPDownError)
        if status is Status.WRONG_SERVER_CLASS:
            return Exception.__new__(WrongServerClass)
        if status is Status.AUTH_FAILED:
            return Exception.__new__(BakAuthLoginFailed)
        if status is Status.INTERNAL_NO_QUOTA:
            return Exception.__new__(InternalQuota)
        if status is Status.VDED_AMP_MISSING:
            return Exception.__new__(NoAmpAccount)
        if status is Status.LOOKUP_MISSING:
            return Exception.__new__(LookupMissing)
        if status is Status.AUTH_MIGRATED:
            return Exception.__new__(DedicatedMoved)
        if status is Status.INTERNAL_VPS:
            return Exception.__new__(VpsRestricted)
        return Exception.__new__(cls)

    def __init__(self, *, status: Status = Status.ERROR, data: str):
        super().__init__(data)
        self.status = status

    def __str__(self):
        return f"{self.status.name}: {super().__str__()}"


class BakAuthDown(BakAuthError):
    """Backup Authority did not reply"""

    __module__ = 'bakauth'


class ClusterUnavailable(BakAuthError):
    """Requested cluster is not available to this client"""

    __module__ = 'bakauth'


class AMPDownError(BakAuthError):
    """Raised when bakauth had an issue reaching PowerPanel and
    the client should retry"""

    __module__ = 'bakauth'


class WrongServerClass(BakAuthError):
    """Accessed the wrong API endpoint for the registered server class"""

    __module__ = 'bakauth'


class BakAuthLoginFailed(BakAuthError):
    """User/Pass for bakauth was wrong"""

    __module__ = 'bakauth'


class BakAuthWrongLogin(BakAuthLoginFailed):
    """For shared servers, this is raised if auth technically succeeded,
    but the server authed as the wrong host, potentially due to a failover
    missing the step of replacing auth.json"""

    __module__ = 'bakauth'


class VpsRestricted(BakAuthError):
    """Access denied to an internal vps"""

    __module__ = 'bakauth'


class LookupMissing(BakAuthError):
    """Requested data could not be found in Backup Authority's database"""

    __module__ = 'bakauth'


class WrongSharedServer(BakAuthError):
    """Raised when calling get_user_bucket() on a user that doesn't
    belong to the server according to amp (or doesn't exist in amp)"""

    __module__ = 'bakauth'


class NoAmpAccount(BakAuthError):
    """Raised when there's no amp account for a v/ded server"""

    __module__ = 'bakauth'


class DedicatedMoved(BakAuthError):
    """This dedicated server's bucket was moved to a new server"""

    __module__ = 'bakauth'


class InternalQuota(BakAuthError):
    """Raised when trying to get quotas for an internal server"""

    __module__ = 'bakauth'


class Unregistered(RuntimeError):
    """Raised when auth.json is missing"""

    __module__ = 'bakauth'
